/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.util;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import net.sf.picard.PicardException;
import net.sf.picard.cmdline.CommandLineParser;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.io.IoUtil;
import net.sf.picard.util.Interval;
import net.sf.picard.util.IntervalList;
import net.sf.picard.util.Log;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMProgramRecord;

public class IntervalListTools
extends CommandLineProgram {
    @Usage
    public final String USAGE = this.getStandardUsagePreamble() + " General tool for manipulating interval lists, " + "including sorting, merging, padding, uniqueifying, and other set-theoretic operations. Default operation if given one or more inputs is to " + "merge and sort them.  Other options are controlled by arguments.";
    @Option(shortName="I", doc="One or more interval lists. If multiple interval lists are provided the output is theresult of merging the inputs.", minElements=1)
    public List<File> INPUT;
    @Option(doc="The output interval list file to write (if SCATTER_COUNT is 1) or the directory into which to write the scattered interval sub-directories (if SCATTER_COUNT > 1)", shortName="O", optional=true)
    public File OUTPUT;
    @Option(doc="The amount to pad each end of the intervals by before other operations are undertaken. Negative numbers are allowed and indicate intervals should be shrunk. Resulting intervals < 0 bases long will be removed. Padding is applied to the interval lists <b> before </b> the ACTION is performed.", optional=true)
    public int PADDING = 0;
    @Option(doc="If true, merge overlapping and adjacent intervals to create a list of unique intervals. Implies SORT=true")
    public boolean UNIQUE = false;
    @Option(doc="If true, sort the resulting interval list by coordinate.")
    public boolean SORT = true;
    @Option(doc="Action to take on inputs.")
    public Action ACTION = Action.CONCAT;
    @Option(shortName="SI", doc="Second set of intervals for SUBTRACT and DIFFERENCE operations.", optional=true)
    public List<File> SECOND_INPUT;
    @Option(doc="One or more lines of comment to add to the header of the output file.", optional=true)
    public List<String> COMMENT = null;
    @Option(doc="The number of files into which to scatter the resulting list by locus.")
    public int SCATTER_COUNT = 1;
    @Option(doc="Produce the inverse list", optional=true)
    public boolean INVERT = false;
    private static final Log log = Log.getInstance(IntervalListTools.class);

    public static void main(String[] args) {
        new IntervalListTools().instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        for (File f : this.INPUT) {
            IoUtil.assertFileIsReadable(f);
        }
        for (File f : this.SECOND_INPUT) {
            IoUtil.assertFileIsReadable(f);
        }
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                IoUtil.assertFileIsWritable(this.OUTPUT);
            } else {
                IoUtil.assertDirectoryIsWritable(this.OUTPUT);
            }
        }
        ArrayList<IntervalList> lists = new ArrayList<IntervalList>();
        for (File f : this.INPUT) {
            IntervalList list = IntervalList.fromFile(f);
            if (this.PADDING != 0) {
                IntervalList out = new IntervalList(list.getHeader());
                for (Interval i : list) {
                    int end;
                    int start = i.getStart() - this.PADDING;
                    if (start > (end = i.getEnd() + this.PADDING)) continue;
                    Interval i2 = new Interval(i.getSequence(), start, end, i.isNegativeStrand(), i.getName());
                    out.add(i2);
                }
                lists.add(out);
                continue;
            }
            lists.add(list);
        }
        ArrayList<IntervalList> secondLists = new ArrayList<IntervalList>();
        for (File f : this.SECOND_INPUT) {
            IntervalList list = IntervalList.fromFile(f);
            if (this.PADDING != 0) {
                IntervalList out = new IntervalList(list.getHeader());
                for (Interval i : list) {
                    int end;
                    int start = i.getStart() - this.PADDING;
                    if (start > (end = i.getEnd() + this.PADDING)) continue;
                    Interval i2 = new Interval(i.getSequence(), start, end, i.isNegativeStrand(), i.getName());
                    out.add(i2);
                }
                secondLists.add(out);
                continue;
            }
            secondLists.add(list);
        }
        if (this.UNIQUE && !this.SORT) {
            log.warn("UNIQUE=true requires sorting but SORT=false was specified.  Results will be sorted!");
        }
        IntervalList result = this.ACTION.act(lists, secondLists);
        if (this.INVERT) {
            this.SORT = false;
            this.UNIQUE = false;
        }
        IntervalList possiblySortedResult = this.SORT ? result.sorted() : result;
        IntervalList possiblyInvertedResult = this.INVERT ? IntervalList.invert(possiblySortedResult) : possiblySortedResult;
        List<Interval> finalIntervals = this.UNIQUE ? possiblyInvertedResult.uniqued().getIntervals() : possiblyInvertedResult.getIntervals();
        SAMFileHeader header = result.getHeader();
        HashSet<String> pgs = new HashSet<String>();
        for (SAMProgramRecord pg : header.getProgramRecords()) {
            pgs.add(pg.getId());
        }
        for (int i = 1; i < Integer.MAX_VALUE; ++i) {
            if (pgs.contains(String.valueOf(i))) continue;
            SAMProgramRecord pg = new SAMProgramRecord(String.valueOf(i));
            pg.setCommandLine(this.getCommandLine());
            pg.setProgramName(this.getClass().getSimpleName());
            header.addProgramRecord(pg);
            break;
        }
        if (this.COMMENT != null) {
            for (String comment : this.COMMENT) {
                header.addComment(comment);
            }
        }
        IntervalList output = new IntervalList(header);
        long total = 0L;
        for (Interval i : finalIntervals) {
            output.add(i);
            total += (long)i.length();
        }
        int intervals = 0;
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                intervals = output.size();
                output.write(this.OUTPUT);
            } else {
                intervals = this.scatterIntervals(output);
            }
        }
        log.info("Output " + intervals + " intervals totalling " + output.getUniqueBaseCount() + " unique bases.");
        return 0;
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.SCATTER_COUNT < 1) {
            return new String[]{"SCATTER_COUNT must be greater than 0."};
        }
        return null;
    }

    private int scatterIntervals(IntervalList list) {
        IntervalList uniquedList = list.uniqued();
        long idealSplitLength = Math.max((long)Math.floor((double)uniquedList.getBaseCount() / (1.0 * (double)this.SCATTER_COUNT)), 1L);
        int splitLength = 0;
        IntervalList split = new IntervalList(uniquedList.getHeader());
        int index = 1;
        Iterator<Interval> it = uniquedList.iterator();
        int totalIntervals = 0;
        DecimalFormat format = new DecimalFormat("0000");
        while (it.hasNext() && index < this.SCATTER_COUNT) {
            Interval interval = it.next();
            int projectedSize = splitLength + interval.length();
            if ((long)projectedSize < idealSplitLength) {
                split.add(interval);
                ++totalIntervals;
                splitLength += interval.length();
                continue;
            }
            if ((long)projectedSize == idealSplitLength) {
                split.add(interval);
                ++totalIntervals;
                split.write(this.createDirectoryAndGetScatterFile(format.format(index++)));
                split = new IntervalList(uniquedList.getHeader());
                splitLength = 0;
                continue;
            }
            int consumed = 0;
            while ((long)projectedSize > idealSplitLength && index < this.SCATTER_COUNT) {
                int amountToConsume = (int)(idealSplitLength - (long)splitLength);
                Interval partial = new Interval(interval.getSequence(), interval.getStart() + consumed, interval.getStart() + consumed + amountToConsume - 1, interval.isNegativeStrand(), interval.getName());
                split.add(partial);
                ++totalIntervals;
                split.write(this.createDirectoryAndGetScatterFile(format.format(index++)));
                split = new IntervalList(uniquedList.getHeader());
                splitLength = 0;
                projectedSize = interval.length() - (consumed += amountToConsume);
            }
            if (projectedSize <= 0) continue;
            Interval remainder = new Interval(interval.getSequence(), interval.getStart() + consumed, interval.getEnd(), interval.isNegativeStrand(), interval.getName());
            split.add(remainder);
            ++totalIntervals;
            splitLength = remainder.length();
        }
        while (it.hasNext()) {
            split.add(it.next());
            ++totalIntervals;
        }
        split.write(this.createDirectoryAndGetScatterFile(format.format(index)));
        return totalIntervals;
    }

    public static File getScatteredFileName(File scatterDirectory, int scatterTotal, String formattedIndex) {
        return new File(scatterDirectory.getAbsolutePath() + "/temp_" + formattedIndex + "_of_" + scatterTotal + "/scattered.intervals");
    }

    private File createDirectoryAndGetScatterFile(String formattedIndex) {
        this.createDirectoryOrFail(this.OUTPUT);
        File result = IntervalListTools.getScatteredFileName(this.OUTPUT, this.SCATTER_COUNT, formattedIndex);
        this.createDirectoryOrFail(result.getParentFile());
        return result;
    }

    private void createDirectoryOrFail(File directory) {
        if (!directory.exists() && !directory.mkdir()) {
            throw new PicardException("Unable to create directory: " + directory.getAbsolutePath());
        }
    }

    public static enum Action implements CommandLineParser.ClpEnum
    {
        CONCAT("The concatenation of all the INPUTs, no sorting or merging of overlapping/abutting intervals implied. Will result in an unsorted list unless requested otherwise."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.concatenate(list);
            }
        }
        ,
        UNION("Like CONCATENATE but with UNIQUE and SORT implied, the result being the set-wise union of all INPUTS."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.union(list);
            }
        }
        ,
        INTERSECT("The sorted, uniqued set of all loci that are contained in all of the INPUTs."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.intersection(list);
            }
        }
        ,
        SUBTRACT("Subtracts SECOND_INPUT from INPUT. The resulting loci are there in INPUT that are not in SECOND_INPUT"){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.subtract(list1, list2);
            }
        }
        ,
        SYMDIFF("Find loci that are in INPUT or SECOND_INPUT but are not in both."){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.difference(list1, list2);
            }
        };

        String helpdoc;

        private Action(String helpdoc) {
            this.helpdoc = helpdoc;
        }

        @Override
        public String getHelpDoc() {
            return this.helpdoc;
        }

        abstract IntervalList act(List<IntervalList> var1, List<IntervalList> var2);
    }
}

