/*
 * Decompiled with CFR 0.152.
 */
package org.broad.tribble.index.tabix;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import net.sf.samtools.Bin;
import net.sf.samtools.BinningIndexContent;
import net.sf.samtools.Chunk;
import net.sf.samtools.LinearIndex;
import net.sf.samtools.util.BlockCompressedInputStream;
import net.sf.samtools.util.BlockCompressedOutputStream;
import net.sf.samtools.util.CloserUtil;
import net.sf.samtools.util.StringUtil;
import org.broad.tribble.TribbleException;
import org.broad.tribble.index.Block;
import org.broad.tribble.index.Index;
import org.broad.tribble.index.tabix.TabixFormat;
import org.broad.tribble.util.LittleEndianInputStream;
import org.broad.tribble.util.LittleEndianOutputStream;

public class TabixIndex
implements Index {
    private static final byte[] MAGIC = new byte[]{84, 66, 73, 1};
    public static final int MAGIC_NUMBER;
    private final TabixFormat formatSpec;
    private final List<String> sequenceNames;
    private final BinningIndexContent[] indices;

    public TabixIndex(TabixFormat formatSpec, List<String> sequenceNames, BinningIndexContent[] indices) {
        if (sequenceNames.size() != indices.length) {
            throw new IllegalArgumentException("sequenceNames.size() != indices.length");
        }
        this.formatSpec = formatSpec.clone();
        this.sequenceNames = Collections.unmodifiableList(new ArrayList<String>(sequenceNames));
        this.indices = indices;
    }

    public TabixIndex(InputStream inputStream) throws IOException {
        this(inputStream, false);
    }

    public TabixIndex(File tabixFile) throws IOException {
        this(new BlockCompressedInputStream(tabixFile), true);
    }

    private TabixIndex(InputStream inputStream, boolean closeInputStream) throws IOException {
        int i;
        LittleEndianInputStream dis = new LittleEndianInputStream(inputStream);
        if (dis.readInt() != MAGIC_NUMBER) {
            throw new TribbleException(String.format("Unexpected magic number 0x%x", MAGIC_NUMBER));
        }
        int numSequences = dis.readInt();
        this.indices = new BinningIndexContent[numSequences];
        this.formatSpec = new TabixFormat();
        this.formatSpec.flags = dis.readInt();
        this.formatSpec.sequenceColumn = dis.readInt();
        this.formatSpec.startPositionColumn = dis.readInt();
        this.formatSpec.endPositionColumn = dis.readInt();
        this.formatSpec.metaCharacter = (char)dis.readInt();
        this.formatSpec.numHeaderLinesToSkip = dis.readInt();
        int nameBlockSize = dis.readInt();
        byte[] nameBlock = new byte[nameBlockSize];
        if (dis.read(nameBlock) != nameBlockSize) {
            throw new EOFException("Premature end of file reading Tabix header");
        }
        ArrayList<String> sequenceNames = new ArrayList<String>(numSequences);
        int startPos = 0;
        for (i = 0; i < numSequences; ++i) {
            int endPos = startPos;
            while (nameBlock[endPos] != 0) {
                ++endPos;
            }
            sequenceNames.add(StringUtil.bytesToString(nameBlock, startPos, endPos - startPos));
            startPos = endPos + 1;
        }
        if (startPos != nameBlockSize) {
            throw new TribbleException("Tabix header format exception.  Sequence name block is longer than expected");
        }
        for (i = 0; i < numSequences; ++i) {
            this.indices[i] = this.loadSequence(i, dis);
        }
        if (closeInputStream) {
            CloserUtil.close(dis);
        }
        this.sequenceNames = Collections.unmodifiableList(sequenceNames);
    }

    @Override
    public List<Block> getBlocks(String chr, int start, int end) {
        int sequenceIndex = this.sequenceNames.indexOf(chr);
        if (sequenceIndex == -1 || this.indices[sequenceIndex] == null) {
            return Collections.EMPTY_LIST;
        }
        List<Chunk> chunks = this.indices[sequenceIndex].getChunksOverlapping(start, end);
        ArrayList<Block> ret = new ArrayList<Block>(chunks.size());
        for (Chunk chunk : chunks) {
            ret.add(new Block(chunk.getChunkStart(), chunk.getChunkEnd() - chunk.getChunkStart()));
        }
        return ret;
    }

    @Override
    public boolean isCurrentVersion() {
        return true;
    }

    @Override
    public List<String> getSequenceNames() {
        return this.sequenceNames;
    }

    @Override
    public boolean containsChromosome(String chr) {
        return this.sequenceNames.contains(chr);
    }

    @Override
    public Map<String, String> getProperties() {
        return null;
    }

    @Override
    public boolean equalsIgnoreProperties(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TabixIndex that = (TabixIndex)o;
        if (!this.formatSpec.equals(that.formatSpec)) {
            return false;
        }
        if (!Arrays.equals(this.indices, that.indices)) {
            return false;
        }
        return ((Object)this.sequenceNames).equals(that.sequenceNames);
    }

    public TabixFormat getFormatSpec() {
        return this.formatSpec;
    }

    public void write(File tabixFile) {
        LittleEndianOutputStream los = new LittleEndianOutputStream(new BlockCompressedOutputStream(tabixFile));
        try {
            this.write(los);
            los.close();
        }
        catch (IOException e) {
            throw new TribbleException("Exception writing " + tabixFile.getAbsolutePath(), e);
        }
    }

    @Override
    public void writeBasedOnFeatureFile(File featureFile) throws IOException {
        if (!featureFile.isFile()) {
            return;
        }
        this.write(new File(featureFile.getAbsolutePath() + ".tbi"));
    }

    @Override
    public void write(LittleEndianOutputStream los) throws IOException {
        los.writeInt(MAGIC_NUMBER);
        los.writeInt(this.sequenceNames.size());
        los.writeInt(this.formatSpec.flags);
        los.writeInt(this.formatSpec.sequenceColumn);
        los.writeInt(this.formatSpec.startPositionColumn);
        los.writeInt(this.formatSpec.endPositionColumn);
        los.writeInt(this.formatSpec.metaCharacter);
        los.writeInt(this.formatSpec.numHeaderLinesToSkip);
        int nameBlockSize = this.sequenceNames.size();
        for (String sequenceName : this.sequenceNames) {
            nameBlockSize += sequenceName.length();
        }
        los.writeInt(nameBlockSize);
        for (String sequenceName : this.sequenceNames) {
            los.write(StringUtil.stringToBytes(sequenceName));
            los.write(0);
        }
        for (BinningIndexContent index : this.indices) {
            this.writeSequence(index, los);
        }
    }

    private void writeSequence(BinningIndexContent indexContent, LittleEndianOutputStream los) throws IOException {
        if (indexContent == null) {
            los.writeInt(0);
        } else {
            BinningIndexContent.BinList binList = indexContent.getBins();
            los.writeInt(binList.numberOfNonNullBins);
            for (Bin bin : binList) {
                this.writeBin(bin, los);
            }
            this.writeLinearIndex(indexContent.getLinearIndex(), los);
        }
    }

    private void writeLinearIndex(LinearIndex linearIndex, LittleEndianOutputStream los) throws IOException {
        if (linearIndex.getIndexStart() != 0) {
            throw new IllegalArgumentException("Non-zero linear index start");
        }
        long[] entries = linearIndex.getIndexEntries();
        los.writeInt(entries.length);
        for (long entry : entries) {
            los.writeLong(entry);
        }
    }

    private void writeBin(Bin bin, LittleEndianOutputStream los) throws IOException {
        los.writeInt(bin.getBinNumber());
        List<Chunk> chunkList = bin.getChunkList();
        los.writeInt(chunkList.size());
        for (Chunk chunk : chunkList) {
            los.writeLong(chunk.getChunkStart());
            los.writeLong(chunk.getChunkEnd());
        }
    }

    private BinningIndexContent loadSequence(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int numBins = dis.readInt();
        if (numBins == 0) {
            return null;
        }
        int nonNullBins = 0;
        ArrayList<Bin> bins = new ArrayList<Bin>();
        for (int i = 0; i < numBins; ++i) {
            Bin bin = this.loadBin(referenceSequenceIndex, dis);
            if (bin == null) continue;
            ++nonNullBins;
            if (bins.size() > bin.getBinNumber()) {
                if (bins.get(bin.getBinNumber()) != null) {
                    throw new TribbleException("Bin " + bin.getBinNumber() + " appears more than once in file");
                }
                bins.set(bin.getBinNumber(), bin);
                continue;
            }
            bins.ensureCapacity(bin.getBinNumber() + 1);
            while (bins.size() < bin.getBinNumber()) {
                bins.add(null);
            }
            bins.add(bin);
        }
        LinearIndex linearIndex = this.loadLinearIndex(referenceSequenceIndex, dis);
        return new BinningIndexContent(referenceSequenceIndex, new BinningIndexContent.BinList(bins.toArray(new Bin[bins.size()]), nonNullBins), linearIndex);
    }

    private LinearIndex loadLinearIndex(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int numElements = dis.readInt();
        long[] elements = new long[numElements];
        for (int i = 0; i < numElements; ++i) {
            elements[i] = dis.readLong();
        }
        return new LinearIndex(referenceSequenceIndex, 0, elements);
    }

    private Bin loadBin(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int binNumber = dis.readInt();
        Bin ret = new Bin(referenceSequenceIndex, binNumber);
        int numChunks = dis.readInt();
        ArrayList<Chunk> chunkList = new ArrayList<Chunk>(numChunks);
        for (int i = 0; i < numChunks; ++i) {
            chunkList.add(this.loadChunk(dis));
        }
        ret.setChunkList(chunkList);
        return ret;
    }

    private Chunk loadChunk(LittleEndianInputStream dis) throws IOException {
        long start = dis.readLong();
        long end = dis.readLong();
        return new Chunk(start, end);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TabixIndex index = (TabixIndex)o;
        if (!this.formatSpec.equals(index.formatSpec)) {
            return false;
        }
        if (!Arrays.equals(this.indices, index.indices)) {
            return false;
        }
        return ((Object)this.sequenceNames).equals(index.sequenceNames);
    }

    public int hashCode() {
        int result = this.formatSpec.hashCode();
        result = 31 * result + ((Object)this.sequenceNames).hashCode();
        result = 31 * result + Arrays.hashCode(this.indices);
        return result;
    }

    static {
        ByteBuffer bb = ByteBuffer.allocate(MAGIC.length);
        bb.put(MAGIC);
        bb.flip();
        MAGIC_NUMBER = bb.order(ByteOrder.LITTLE_ENDIAN).getInt();
    }
}

